# (c) Copyright 2009-2013. CodeWeavers, Inc.

import traceback

from gi.repository import Gtk

import bottlequery
import cxguitools
import pyop

import appdetector
import c4profilesmanager

import cxlog
import cxutils

# for localization
from cxutils import cxgettext as _


class AssocEditDialog(object):
    def __init__(self, bottlename):
        self.bottlename = bottlename

        # Import widget modules so Gtk.Builder() finds them
        import cxassocwidget # pylint: disable=W0612,W0611
        self.xml = Gtk.Builder()
        self.xml.set_translation_domain("crossover")
        self.xml.add_from_file(cxguitools.get_ui_path("cxassocedit"))
        self.xml.connect_signals(self)

        self.dialog = self.xml.get_object('assoc-edit-dialog')
        self.edit_widget = self.xml.get_object('assoc-editor')
        self.common_only_checkbox = self.xml.get_object('common-only')

        self.dialog.set_title(_('Associations in %(bottlename)s') % {'bottlename':bottlename})

        properties = bottlequery.get_bottle_properties(bottlename)
        if properties['managed']:
            self.xml.get_object("recreate_button").hide()
            self.xml.get_object("apply_button").hide()
            self.xml.get_object("cancel_button").hide()
            self.xml.get_object("ok_button").hide()
        else:
            self.xml.get_object("close_button").hide()

        self.edit_widget.set_bottle(bottlename, properties['managed'])

        self.edit_widget.refresh(self.enable_buttons, self.refresh_fail)

        self.edit_widget.show() # FIXME: Why is this necessary?
        self.dialog.show()

        self.common_eassocs = set()

        self.edit_widget.set_filter(self.assoc_is_common)

        # Get the list of installed applications so we know which
        # associations are "common".
        operation = GetInstalledAppsOp(self)
        pyop.sharedOperationQueue.enqueue(operation)

    def got_installed_apps(self, apps):
        self.common_only_checkbox.set_sensitive(True)
        for app in cxutils.itervalues(apps):
            profile = app.profile
            if profile:
                # Ignore use_if and take all the installer profiles into
                # account, even if they are not for the current profile!
                for inst_profile in profile.installer_profiles:
                    self.common_eassocs.update(inst_profile.default_eassocs)
                    self.common_eassocs.update(inst_profile.alt_eassocs)
        self.edit_widget.set_filter(self.assoc_is_common)

    def assoc_is_common(self, eassoc):
        return eassoc in self.common_eassocs

    def on_common_only_toggled(self, togglebutton):
        if togglebutton.get_active():
            self.edit_widget.set_filter(self.assoc_is_common)
        else:
            self.edit_widget.set_filter(None)

    def disable_buttons(self):
        self.xml.get_object("recreate_button").set_sensitive(False)
        self.xml.get_object("apply_button").set_sensitive(False)
        self.xml.get_object("cancel_button").set_sensitive(False)
        self.xml.get_object("ok_button").set_sensitive(False)

    def enable_buttons(self):
        self.xml.get_object("recreate_button").set_sensitive(True)
        self.xml.get_object("apply_button").set_sensitive(True)
        self.xml.get_object("cancel_button").set_sensitive(True)
        self.xml.get_object("ok_button").set_sensitive(True)

    def ok_finished(self):
        self.dialog.destroy()

    def apply_fail(self, error):
        self.enable_buttons()
        cxguitools.CXMessageDlg(
            primary=_("Could not save changes to associations"),
            secondary=error,
            parent=self.dialog,
            message_type=Gtk.MessageType.ERROR)

    def quit_message_box(self, _response):
        self.dialog.destroy()

    def refresh_fail(self, error):
        self.dialog.hide()
        cxguitools.CXMessageDlg(
            primary=_("Could not read associations for %(bottlename)s") % {'bottlename': self.bottlename},
            secondary=error,
            parent=self.dialog, response_function=self.quit_message_box,
            message_type=Gtk.MessageType.ERROR)

    def on_recreate_button_clicked(self, _button):
        self.disable_buttons()
        self.edit_widget.recreate_assocs(self.enable_buttons, self.apply_fail)

    def on_apply_button_clicked(self, _button):
        self.disable_buttons()
        self.edit_widget.commit(self.enable_buttons, self.apply_fail)

    def on_cancel_button_clicked(self, _button):
        self.dialog.destroy()

    def on_ok_button_clicked(self, _button):
        self.disable_buttons()
        self.edit_widget.commit(self.ok_finished, self.apply_fail)

    def on_delete(self, _widget, _event):
        self.dialog.destroy()

    def quit_requested(self, _caller):
        # pylint: disable=R0201
        cxguitools.toplevel_quit()

class GetInstalledAppsOp(pyop.PythonOperation):
    def __init__(self, dialog):
        pyop.PythonOperation.__init__(self)
        self.dialog = dialog
        self.installed_apps = None

    def main(self):
        self.installed_apps = None
        try:
            self.installed_apps = appdetector.fast_get_installed_applications(
                self.dialog.bottlename, c4profilesmanager.profilesSet())
        except Exception: # pylint: disable=W0703
            cxlog.warn("Error getting installed apps: \n%s" % traceback.format_exc())

    def finish(self):
        if self.installed_apps is not None:
            self.dialog.got_installed_apps(self.installed_apps)

def start(bottle):
    AssocEditDialog(bottle.name)
