/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.util;

import java.io.IOException;
import java.io.InputStream;
import java.lang.invoke.MethodHandles;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Base64;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import org.apache.solr.common.SolrException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class CryptoKeys {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final String CIPHER_ALGORITHM = "RSA/ECB/nopadding";
    private final Map<String, PublicKey> keys;
    private Exception exception;

    public CryptoKeys(Map<String, byte[]> trustedKeys) throws Exception {
        HashMap<String, PublicKey> m = new HashMap<String, PublicKey>();
        for (Map.Entry<String, byte[]> e : trustedKeys.entrySet()) {
            m.put(e.getKey(), CryptoKeys.getX509PublicKey(e.getValue()));
        }
        this.keys = Map.copyOf(m);
    }

    public String verify(String sig, ByteBuffer data) {
        this.exception = null;
        for (Map.Entry<String, PublicKey> entry : this.keys.entrySet()) {
            try {
                boolean verified = CryptoKeys.verify(entry.getValue(), Base64.getDecoder().decode(sig), data);
                log.debug("verified {} ", (Object)verified);
                if (!verified) continue;
                return entry.getKey();
            }
            catch (Exception e) {
                this.exception = e;
                log.debug("NOT verified  ");
            }
        }
        return null;
    }

    public String verify(String sig, InputStream is) {
        this.exception = null;
        for (Map.Entry<String, PublicKey> entry : this.keys.entrySet()) {
            try {
                boolean verified = CryptoKeys.verify(entry.getValue(), Base64.getDecoder().decode(sig), is);
                log.debug("verified {} ", (Object)verified);
                if (!verified) continue;
                return entry.getKey();
            }
            catch (Exception e) {
                this.exception = e;
                log.debug("NOT verified  ");
            }
        }
        return null;
    }

    public static PublicKey getX509PublicKey(byte[] buf) throws InvalidKeySpecException {
        X509EncodedKeySpec spec = new X509EncodedKeySpec(buf);
        try {
            KeyFactory kf = KeyFactory.getInstance("RSA");
            return kf.generatePublic(spec);
        }
        catch (NoSuchAlgorithmException e) {
            throw new AssertionError("JVM spec is required to support RSA", e);
        }
    }

    public static boolean verify(PublicKey publicKey, byte[] sig, ByteBuffer data) throws InvalidKeyException, SignatureException {
        data = ByteBuffer.wrap(data.array(), data.arrayOffset(), data.limit());
        try {
            Signature signature = Signature.getInstance("SHA1withRSA");
            signature.initVerify(publicKey);
            signature.update(data);
            return signature.verify(sig);
        }
        catch (NoSuchAlgorithmException e) {
            throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, (Throwable)e);
        }
    }

    public static boolean verify(PublicKey publicKey, byte[] sig, InputStream is) throws InvalidKeyException, SignatureException, IOException {
        try {
            int sz;
            Signature signature = Signature.getInstance("SHA1withRSA");
            signature.initVerify(publicKey);
            byte[] buf = new byte[1024];
            while ((sz = is.read(buf)) != -1) {
                signature.update(buf, 0, sz);
            }
            try {
                return signature.verify(sig);
            }
            catch (SignatureException e) {
                return false;
            }
        }
        catch (NoSuchAlgorithmException e) {
            throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, (Throwable)e);
        }
    }

    public static PublicKey deserializeX509PublicKey(String pubKey) {
        try {
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            X509EncodedKeySpec publicKeySpec = new X509EncodedKeySpec(Base64.getDecoder().decode(pubKey));
            return keyFactory.generatePublic(publicKeySpec);
        }
        catch (Exception e) {
            throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, (Throwable)e);
        }
    }

    public static byte[] decryptRSA(byte[] buffer, PublicKey pubKey) throws InvalidKeyException, BadPaddingException, IllegalBlockSizeException {
        Cipher rsaCipher;
        try {
            rsaCipher = Cipher.getInstance(CIPHER_ALGORITHM);
        }
        catch (Exception e) {
            throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, (Throwable)e);
        }
        rsaCipher.init(2, pubKey);
        return rsaCipher.doFinal(buffer);
    }

    public static boolean verifySha256(byte[] data, byte[] sig, PublicKey key) throws SignatureException, InvalidKeyException {
        try {
            Signature signature = Signature.getInstance("SHA256withRSA");
            signature.initVerify(key);
            signature.update(data);
            return signature.verify(sig);
        }
        catch (NoSuchAlgorithmException e) {
            throw new InternalError("SHA256withRSA must be supported by the JVM.");
        }
    }

    public static Collection<X509Certificate> parseX509Certs(InputStream certsStream) {
        try {
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            Collection<? extends Certificate> parsedCerts = cf.generateCertificates(certsStream);
            List<X509Certificate> certs = parsedCerts.stream().filter(c -> c instanceof X509Certificate).map(c -> (X509Certificate)c).collect(Collectors.toList());
            if (certs.size() > 0) {
                return certs;
            }
            throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Wrong type of certificates. Must be DER or PEM format");
        }
        catch (CertificateException e) {
            throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Failed loading certificate(s) from input stream", (Throwable)e);
        }
    }

    public static String extractCertificateFromPem(String pemContents) {
        int from = pemContents.indexOf("-----BEGIN CERTIFICATE-----");
        int end = pemContents.lastIndexOf("-----END CERTIFICATE-----") + 25;
        return pemContents.substring(from, end);
    }

    public static class RSAKeyPair {
        private final String pubKeyStr;
        private final PublicKey publicKey;
        private final PrivateKey privateKey;
        private static final int DEFAULT_KEYPAIR_LENGTH = 2048;
        private final int keySizeInBytes;

        public RSAKeyPair() {
            KeyPairGenerator keyGen;
            try {
                keyGen = KeyPairGenerator.getInstance("RSA");
            }
            catch (NoSuchAlgorithmException e) {
                throw new AssertionError("JVM spec is required to support RSA", e);
            }
            keyGen.initialize(2048);
            KeyPair keyPair = keyGen.genKeyPair();
            this.privateKey = keyPair.getPrivate();
            this.keySizeInBytes = this.determineKeySizeInBytes(this.privateKey);
            this.publicKey = keyPair.getPublic();
            this.pubKeyStr = Base64.getEncoder().encodeToString(this.publicKey.getEncoded());
        }

        public RSAKeyPair(URL privateKeyResourceName, URL publicKeyResourceName) throws IOException, InvalidKeySpecException {
            try (InputStream inPrivate = privateKeyResourceName.openStream();){
                String privateString = new String(inPrivate.readAllBytes(), StandardCharsets.UTF_8).replaceAll("-----(BEGIN|END) PRIVATE KEY-----", "");
                PKCS8EncodedKeySpec privateSpec = new PKCS8EncodedKeySpec(Base64.getMimeDecoder().decode(privateString));
                KeyFactory rsaFactory = KeyFactory.getInstance("RSA");
                this.privateKey = rsaFactory.generatePrivate(privateSpec);
                this.keySizeInBytes = this.determineKeySizeInBytes(this.privateKey);
            }
            catch (NoSuchAlgorithmException e) {
                throw new AssertionError("JVM spec is required to support RSA", e);
            }
            try (InputStream inPublic = publicKeyResourceName.openStream();){
                this.publicKey = CryptoKeys.getX509PublicKey(inPublic.readAllBytes());
                this.pubKeyStr = Base64.getEncoder().encodeToString(this.publicKey.getEncoded());
            }
        }

        public String getPublicKeyStr() {
            return this.pubKeyStr;
        }

        public PublicKey getPublicKey() {
            return this.publicKey;
        }

        private int determineKeySizeInBytes(PrivateKey privateKey) {
            return ((RSAPrivateKey)privateKey).getModulus().bitLength() / 8;
        }

        public int getKeySizeInBytes() {
            return this.keySizeInBytes;
        }

        public byte[] encrypt(ByteBuffer buffer) {
            byte[] paddedPlaintext = new byte[this.getKeySizeInBytes()];
            buffer.get(paddedPlaintext, buffer.arrayOffset() + buffer.position(), buffer.limit());
            try {
                Cipher rsaCipher = Cipher.getInstance(CryptoKeys.CIPHER_ALGORITHM);
                rsaCipher.init(1, this.privateKey);
                return rsaCipher.doFinal(paddedPlaintext);
            }
            catch (Exception e) {
                throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, (Throwable)e);
            }
        }

        public byte[] signSha256(byte[] bytes) {
            Signature dsa = null;
            try {
                dsa = Signature.getInstance("SHA256withRSA");
            }
            catch (NoSuchAlgorithmException e) {
                throw new InternalError("SHA256withRSA is required to be supported by the JVM.", e);
            }
            try {
                dsa.initSign(this.privateKey);
                dsa.update(bytes, 0, bytes.length);
                return dsa.sign();
            }
            catch (InvalidKeyException | SignatureException e) {
                throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Error generating PKI Signature", (Throwable)e);
            }
        }
    }
}

