﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/redshift/Redshift_EXPORTS.h>
#include <aws/redshift/model/Tag.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace Redshift {
namespace Model {

/**
 * <p>The snapshot copy grant that grants Amazon Redshift permission to encrypt
 * copied snapshots with the specified encrypted symmetric key from Amazon Web
 * Services KMS in the destination region.</p> <p> For more information about
 * managing snapshot copy grants, go to <a
 * href="https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-db-encryption.html">Amazon
 * Redshift Database Encryption</a> in the <i>Amazon Redshift Cluster Management
 * Guide</i>. </p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/redshift-2012-12-01/SnapshotCopyGrant">AWS
 * API Reference</a></p>
 */
class SnapshotCopyGrant {
 public:
  AWS_REDSHIFT_API SnapshotCopyGrant() = default;
  AWS_REDSHIFT_API SnapshotCopyGrant(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_REDSHIFT_API SnapshotCopyGrant& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_REDSHIFT_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_REDSHIFT_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The name of the snapshot copy grant.</p>
   */
  inline const Aws::String& GetSnapshotCopyGrantName() const { return m_snapshotCopyGrantName; }
  inline bool SnapshotCopyGrantNameHasBeenSet() const { return m_snapshotCopyGrantNameHasBeenSet; }
  template <typename SnapshotCopyGrantNameT = Aws::String>
  void SetSnapshotCopyGrantName(SnapshotCopyGrantNameT&& value) {
    m_snapshotCopyGrantNameHasBeenSet = true;
    m_snapshotCopyGrantName = std::forward<SnapshotCopyGrantNameT>(value);
  }
  template <typename SnapshotCopyGrantNameT = Aws::String>
  SnapshotCopyGrant& WithSnapshotCopyGrantName(SnapshotCopyGrantNameT&& value) {
    SetSnapshotCopyGrantName(std::forward<SnapshotCopyGrantNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the encrypted symmetric key in Amazon Web Services
   * KMS to which Amazon Redshift is granted permission.</p>
   */
  inline const Aws::String& GetKmsKeyId() const { return m_kmsKeyId; }
  inline bool KmsKeyIdHasBeenSet() const { return m_kmsKeyIdHasBeenSet; }
  template <typename KmsKeyIdT = Aws::String>
  void SetKmsKeyId(KmsKeyIdT&& value) {
    m_kmsKeyIdHasBeenSet = true;
    m_kmsKeyId = std::forward<KmsKeyIdT>(value);
  }
  template <typename KmsKeyIdT = Aws::String>
  SnapshotCopyGrant& WithKmsKeyId(KmsKeyIdT&& value) {
    SetKmsKeyId(std::forward<KmsKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tag instances.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  SnapshotCopyGrant& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  SnapshotCopyGrant& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_snapshotCopyGrantName;
  bool m_snapshotCopyGrantNameHasBeenSet = false;

  Aws::String m_kmsKeyId;
  bool m_kmsKeyIdHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Redshift
}  // namespace Aws
