package Lire::Config::ReportConfigIndex;

use strict;

use base qw/Lire::Config::Index/;

use Lire::Utils qw/check_param check_object_param/;
use Carp;

=pod

=head1 NAME

Lire::Config::ReportConfigIndex - Index for store report configurations.

=head1 SYNOPSIS

  <lrcsml:reference name="name" index="store_report_specs" />

=head1 DESCRIPTION

This object make it possible to select from a list of ReportSpec. Each
time as Lire::DlfStore is opened all the report configurations defined
in it will be available under the 'store_report_specs' Index name.

=head2 new( $list )

Creates a new ReportConfigIndex which wraps the $list report
configurations list. This should be a Lire::Config::List instance with
a spec that contains only one Lire::Config::ReportSpec component.

=cut

sub new {
    my ( $class, $list ) = @_;

    check_object_param( $list, 'list', 'Lire::Config::List' );
    my @names = $list->spec()->component_names();
    croak "'list' spec should contain only one Lire::Config::ReportSpec component"
      unless @names == 1 && $list->spec()->get( $names[0] )->isa( 'Lire::Config::ReportSpec' );

    return bless { '_list' => $list }, $class;
}

sub has_entry {
    my ( $self, $name ) = @_;

    check_param( $name, 'name' );

    foreach my $cfg ( $self->{'_list'}->elements() ) {
        no warnings 'uninitialized';
        return 1 if $cfg->get( 'id' )->get() eq $name;
    }
    return 0;
}

sub entries {
    my $self = $_[0];

    return [ sort map { $_->get( 'id' )->get() } $self->{'_list'}->elements() ];
}

sub get_ref {
    my ( $self, $name ) = @_;

    check_param( $name, 'name' );
    croak "no report configuration \"$name\""
      unless $self->has_entry( $name );

    foreach my $cfg ( $self->{'_list'}->elements() ) {
        no warnings 'uninitialized';
        return $cfg->as_value() if $cfg->get( 'id' )->get() eq $name;
    }
    return;
}

1;

__END__

=pod

=head1 SEE ALSO

 Lire::Config::ReportConfig(3pm) Lire:Config::Index(3pm)

=head1 AUTHOR

  Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: ReportConfigIndex.pm,v 1.2 2006/07/23 13:16:30 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html.

=cut
