// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { needs, KeyringTraceFlag, immutableClass, readOnlyProperty, unwrapDataKey, } from '@aws-crypto/material-management';
import { KMS_PROVIDER_ID, generateDataKey, encrypt, decrypt, kmsResponseToEncryptedDataKey, } from './helpers';
import { mrkAwareAwsKmsKeyIdCompare, parseAwsKmsKeyArn, validAwsKmsIdentifier, } from './arn_parsing';
export function AwsKmsMrkAwareSymmetricKeyringClass(BaseKeyring) {
    class AwsKmsMrkAwareSymmetricKeyring
    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.5
    //# MUST implement the AWS Encryption SDK Keyring interface (../keyring-
    //# interface.md#interface)
     extends BaseKeyring {
        keyId;
        client;
        grantTokens;
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.6
        //# On initialization the caller MUST provide:
        constructor({ client, keyId, grantTokens, }) {
            super();
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.6
            //# The AWS KMS key identifier MUST NOT be null or empty.
            needs(keyId && typeof keyId === 'string', 'An AWS KMS key identifier is required.');
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.6
            //# The AWS KMS
            //# key identifier MUST be a valid identifier (aws-kms-key-arn.md#a-
            //# valid-aws-kms-identifier).
            needs(validAwsKmsIdentifier(keyId), `Key id ${keyId} is not a valid identifier.`);
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.6
            //# The AWS KMS
            //# SDK client MUST NOT be null.
            needs(!!client, 'An AWS SDK client is required');
            readOnlyProperty(this, 'client', client);
            readOnlyProperty(this, 'keyId', keyId);
            readOnlyProperty(this, 'grantTokens', grantTokens);
        }
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
        //# OnEncrypt MUST take encryption materials (structures.md#encryption-
        //# materials) as input.
        async _onEncrypt(material) {
            const { client, keyId, grantTokens } = this;
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
            //# If the input encryption materials (structures.md#encryption-
            //# materials) do not contain a plaintext data key OnEncrypt MUST attempt
            //# to generate a new plaintext data key by calling AWS KMS
            //# GenerateDataKey (https://docs.aws.amazon.com/kms/latest/APIReference/
            //# API_GenerateDataKey.html).
            if (!material.hasUnencryptedDataKey) {
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# The keyring MUST call
                //# AWS KMS GenerateDataKeys with a request constructed as follows:
                //
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# If the call to AWS KMS GenerateDataKey
                //# (https://docs.aws.amazon.com/kms/latest/APIReference/
                //# API_GenerateDataKey.html) does not succeed, OnEncrypt MUST NOT modify
                //# the encryption materials (structures.md#encryption-materials) and
                //# MUST fail.
                const dataKey = await generateDataKey(
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# If the keyring calls AWS KMS GenerateDataKeys, it MUST use the
                //# configured AWS KMS client to make the call.
                client, material.suite.keyLengthBytes, keyId, material.encryptionContext, grantTokens);
                /* This should be impossible given that generateDataKey only returns false if the client supplier does. */
                needs(dataKey, 'Generator KMS key did not generate a data key');
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# The Generate Data Key response's "KeyId" MUST be A valid AWS
                //# KMS key ARN (aws-kms-key-arn.md#identifying-an-aws-kms-multi-region-
                //# key).
                needs(parseAwsKmsKeyArn(dataKey.KeyId), 'Malformed arn.');
                const flags = KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY |
                    KeyringTraceFlag.WRAPPING_KEY_SIGNED_ENC_CTX |
                    KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY;
                const trace = {
                    keyNamespace: KMS_PROVIDER_ID,
                    keyName: dataKey.KeyId,
                    flags,
                };
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# If verified,
                //# OnEncrypt MUST do the following with the response from AWS KMS
                //# GenerateDataKey (https://docs.aws.amazon.com/kms/latest/APIReference/
                //# API_GenerateDataKey.html):
                material
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                    //# If the Generate Data Key call succeeds, OnEncrypt MUST verify that
                    //# the response "Plaintext" length matches the specification of the
                    //# algorithm suite (algorithm-suites.md)'s Key Derivation Input Length
                    //# field.
                    //
                    // setUnencryptedDataKey will throw if the plaintext does not match the algorithm suite requirements.
                    .setUnencryptedDataKey(dataKey.Plaintext, trace)
                    .addEncryptedDataKey(kmsResponseToEncryptedDataKey(dataKey), KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY |
                    KeyringTraceFlag.WRAPPING_KEY_SIGNED_ENC_CTX);
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# *  OnEncrypt MUST output the modified encryption materials
                //# (structures.md#encryption-materials)
                return material;
            }
            else {
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# Given a plaintext data key in the encryption materials
                //# (structures.md#encryption-materials), OnEncrypt MUST attempt to
                //# encrypt the plaintext data key using the configured AWS KMS key
                //# identifier.
                const unencryptedDataKey = unwrapDataKey(material.getUnencryptedDataKey());
                const flags = KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY |
                    KeyringTraceFlag.WRAPPING_KEY_SIGNED_ENC_CTX;
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# If the call to AWS KMS Encrypt
                //# (https://docs.aws.amazon.com/kms/latest/APIReference/
                //# API_Encrypt.html) does not succeed, OnEncrypt MUST fail.
                //
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# The keyring
                //# MUST AWS KMS Encrypt call with a request constructed as follows:
                const kmsEDK = await encrypt(
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# The keyring MUST call AWS KMS Encrypt
                //# (https://docs.aws.amazon.com/kms/latest/APIReference/
                //# API_Encrypt.html) using the configured AWS KMS client.
                client, unencryptedDataKey, keyId, material.encryptionContext, grantTokens);
                /* This should be impossible given that encrypt only returns false if the client supplier does. */
                needs(kmsEDK, 'AwsKmsMrkAwareSymmetricKeyring failed to encrypt data key');
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# If the Encrypt call succeeds The response's "KeyId" MUST be A valid
                //# AWS KMS key ARN (aws-kms-key-arn.md#identifying-an-aws-kms-multi-
                //# region-key).
                needs(parseAwsKmsKeyArn(kmsEDK.KeyId), 'Malformed arn.');
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# If verified, OnEncrypt MUST do the following with the response from
                //# AWS KMS Encrypt (https://docs.aws.amazon.com/kms/latest/APIReference/
                //# API_Encrypt.html):
                material.addEncryptedDataKey(kmsResponseToEncryptedDataKey(kmsEDK), flags);
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# If all Encrypt calls succeed, OnEncrypt MUST output the modified
                //# encryption materials (structures.md#encryption-materials).
                return material;
            }
        }
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
        //# OnDecrypt MUST take decryption materials (structures.md#decryption-
        //# materials) and a list of encrypted data keys
        //# (structures.md#encrypted-data-key) as input.
        async _onDecrypt(material, encryptedDataKeys) {
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
            //# If the decryption materials (structures.md#decryption-materials)
            //# already contained a valid plaintext data key OnDecrypt MUST
            //# immediately return the unmodified decryption materials
            //# (structures.md#decryption-materials).
            if (material.hasValidKey())
                return material;
            const { client, keyId, grantTokens } = this;
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
            //# The set of encrypted data keys MUST first be filtered to match this
            //# keyring's configuration.
            const decryptableEDKs = encryptedDataKeys.filter(filterEDKs(keyId));
            const cmkErrors = [];
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
            //# For each encrypted data key in the filtered set, one at a time, the
            //# OnDecrypt MUST attempt to decrypt the data key.
            for (const edk of decryptableEDKs) {
                const { providerId, encryptedDataKey } = edk;
                try {
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
                    //# When calling AWS KMS Decrypt
                    //# (https://docs.aws.amazon.com/kms/latest/APIReference/
                    //# API_Decrypt.html), the keyring MUST call with a request constructed
                    //# as follows:
                    const dataKey = await decrypt(
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
                    //# To attempt to decrypt a particular encrypted data key
                    //# (structures.md#encrypted-data-key), OnDecrypt MUST call AWS KMS
                    //# Decrypt (https://docs.aws.amazon.com/kms/latest/APIReference/
                    //# API_Decrypt.html) with the configured AWS KMS client.
                    client, 
                    // For MRKs the key identifier MUST be the configured key identifer.
                    { providerId, encryptedDataKey, providerInfo: this.keyId }, material.encryptionContext, grantTokens);
                    /* This should be impossible given that decrypt only returns false if the client supplier does
                     * or if the providerId is not "aws-kms", which we have already filtered out
                     */
                    needs(dataKey, 'decrypt did not return a data key.');
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
                    //# *  The "KeyId" field in the response MUST equal the configured AWS
                    //# KMS key identifier.
                    needs(dataKey.KeyId === this.keyId, 'KMS Decryption key does not match the requested key id.');
                    const flags = KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY |
                        KeyringTraceFlag.WRAPPING_KEY_VERIFIED_ENC_CTX;
                    const trace = {
                        keyNamespace: KMS_PROVIDER_ID,
                        keyName: dataKey.KeyId,
                        flags,
                    };
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
                    //# If the response does satisfies these requirements then OnDecrypt MUST
                    //# do the following with the response:
                    //
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
                    //# *  The length of the response's "Plaintext" MUST equal the key
                    //# derivation input length (algorithm-suites.md#key-derivation-input-
                    //# length) specified by the algorithm suite (algorithm-suites.md)
                    //# included in the input decryption materials
                    //# (structures.md#decryption-materials).
                    //
                    // setUnencryptedDataKey will throw if the plaintext does not match the algorithm suite requirements.
                    material.setUnencryptedDataKey(dataKey.Plaintext, trace);
                    return material;
                }
                catch (e) {
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
                    //# If this attempt
                    //# results in an error, then these errors MUST be collected.
                    //
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
                    //# If the response does not satisfies these requirements then an error
                    //# MUST be collected and the next encrypted data key in the filtered set
                    //# MUST be attempted.
                    cmkErrors.push({ errPlus: e });
                }
            }
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
            //# If OnDecrypt fails to successfully decrypt any encrypted data key
            //# (structures.md#encrypted-data-key), then it MUST yield an error that
            //# includes all the collected errors.
            needs(material.hasValidKey(), [
                `Unable to decrypt data key${!decryptableEDKs.length ? ': No EDKs supplied' : ''}.`,
                ...cmkErrors.map((e, i) => `Error #${i + 1}  \n${e.errPlus.stack}`),
            ].join('\n'));
            return material;
        }
    }
    immutableClass(AwsKmsMrkAwareSymmetricKeyring);
    return AwsKmsMrkAwareSymmetricKeyring;
}
function filterEDKs(keyringKeyId) {
    return function filter({ providerId, providerInfo }) {
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
        //# *  Its provider ID MUST exactly match the value "aws-kms".
        if (providerId !== KMS_PROVIDER_ID)
            return false;
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
        //# *  The provider info MUST be a valid AWS KMS ARN (aws-kms-key-
        //# arn.md#a-valid-aws-kms-arn) with a resource type of "key" or
        //# OnDecrypt MUST fail.
        const arnInfo = parseAwsKmsKeyArn(providerInfo);
        needs(arnInfo && arnInfo.ResourceType === 'key', 'Unexpected EDK ProviderInfo for AWS KMS EDK');
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
        //# *  The the function AWS KMS MRK Match for Decrypt (aws-kms-mrk-match-
        //# for-decrypt.md#implementation) called with the configured AWS KMS
        //# key identifier and the provider info MUST return "true".
        return mrkAwareAwsKmsKeyIdCompare(keyringKeyId, providerInfo);
    };
}
//# sourceMappingURL=data:application/json;base64,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