/****************************************************************************
Copyright 2017  Martin Flöser <mgraesslin@kde.org>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) version 3, or any
later version accepted by the membership of KDE e.V. (or its
successor approved by the membership of KDE e.V.), which shall
act as a proxy defined in Section 6 of version 3 of the license.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library.  If not, see <http://www.gnu.org/licenses/>.
****************************************************************************/
#ifndef WRAPLAND_CLIENT_IDLEINHIBIT_H
#define WRAPLAND_CLIENT_IDLEINHIBIT_H

#include <QObject>
// STD
#include <Wrapland/Client/wraplandclient_export.h>
#include <memory>

struct zwp_idle_inhibit_manager_v1;
struct zwp_idle_inhibitor_v1;

namespace Wrapland
{
namespace Client
{

class EventQueue;
class Surface;
class IdleInhibitor;

/**
 * @short Wrapper for the zwp_idle_inhibit_manager_v1 interface.
 *
 * This class provides a convenient wrapper for the zwp_idle_inhibit_manager_v1 interface.
 *
 * To use this class one needs to interact with the Registry. There are two
 * possible ways to create the IdleInhibitManager interface:
 * @code
 * IdleInhibitManager *c = registry->createIdleInhibitManager(name, version);
 * @endcode
 *
 * This creates the IdleInhibitManager and sets it up directly. As an alternative this
 * can also be done in a more low level way:
 * @code
 * IdleInhibitManager *c = new IdleInhibitManager;
 * c->setup(registry->bindIdleInhibitManager(name, version));
 * @endcode
 *
 * The IdleInhibitManager can be used as a drop-in replacement for any zwp_idle_inhibit_manager_v1
 * pointer as it provides matching cast operators.
 *
 * @see Registry
 * @since 0.0.541
 **/
class WRAPLANDCLIENT_EXPORT IdleInhibitManager : public QObject
{
    Q_OBJECT
public:
    /**
     * Creates a new IdleInhibitManager.
     * Note: after constructing the IdleInhibitManager it is not yet valid and one needs
     * to call setup. In order to get a ready to use IdleInhibitManager prefer using
     * Registry::createIdleInhibitManager.
     **/
    explicit IdleInhibitManager(QObject* parent = nullptr);
    virtual ~IdleInhibitManager();

    /**
     * Setup this IdleInhibitManager to manage the @p idleinhibitmanager.
     * When using Registry::createIdleInhibitManager there is no need to call this
     * method.
     **/
    void setup(zwp_idle_inhibit_manager_v1* idleinhibitmanager);
    /**
     * @returns @c true if managing a zwp_idle_inhibit_manager_v1.
     **/
    bool isValid() const;
    /**
     * Releases the zwp_idle_inhibit_manager_v1 interface.
     * After the interface has been released the IdleInhibitManager instance is no
     * longer valid and can be setup with another zwp_idle_inhibit_manager_v1 interface.
     **/
    void release();

    /**
     * Sets the @p queue to use for creating objects with this IdleInhibitManager.
     **/
    void setEventQueue(EventQueue* queue);
    /**
     * @returns The event queue to use for creating objects with this IdleInhibitManager.
     **/
    EventQueue* eventQueue();

    /**
     * Creates an IdleInhibitor for the given @p surface.
     * While the IdleInhibitor exists the @p surface is marked to inhibit idle.
     * @param surface The Surface which should have idle inhibited
     * @param parent The parent object for the IdleInhibitor
     * @returns The created IdleInhibitor
     **/
    IdleInhibitor* createInhibitor(Surface* surface, QObject* parent = nullptr);

    operator zwp_idle_inhibit_manager_v1*();
    operator zwp_idle_inhibit_manager_v1*() const;

Q_SIGNALS:
    /**
     * The corresponding global for this interface on the Registry got removed.
     *
     * This signal gets only emitted if the IdleInhibitManager got created by
     * Registry::createIdleInhibitManager
     **/
    void removed();

private:
    class Private;
    std::unique_ptr<Private> d;
};

/**
 * An IdleInhibitor prevents the Output that the associated Surface is visible on from being
 * set to a state where it is not visually usable due to lack of user interaction
 * (e.g. blanked, dimmed, locked, set to power save, etc.)  Any screensaver processes are
 * also blocked from displaying.
 *
 * If the Surface is destroyed, unmapped, becomes occluded, loses visibility, or otherwise
 * becomes not visually relevant for the user, the IdleInhibitor will not be honored by
 * the compositor; if the Surface subsequently regains visibility the inhibitor takes effect
 * once again.
 * Likewise, the IdleInhibitor isn't honored if the system was already idled at the time the
 * IdleInhibitor was established, although if the system later de-idles and re-idles the
 * IdleInhibitor will take effect.
 *
 * @see IdleInhibitManager
 * @see Surface
 * @since 0.0.541
 **/
class WRAPLANDCLIENT_EXPORT IdleInhibitor : public QObject
{
    Q_OBJECT
public:
    virtual ~IdleInhibitor();

    /**
     * Setup this IdleInhibitor to manage the @p idleinhibitor.
     * When using IdleInhibitManager::createIdleInhibitor there is no need to call this
     * method.
     **/
    void setup(zwp_idle_inhibitor_v1* idleinhibitor);
    /**
     * @returns @c true if managing a zwp_idle_inhibitor_v1.
     **/
    bool isValid() const;
    /**
     * Releases the zwp_idle_inhibitor_v1 interface.
     * After the interface has been released the IdleInhibitor instance is no
     * longer valid and can be setup with another zwp_idle_inhibitor_v1 interface.
     **/
    void release();
    /**
     * Destroys the data held by this IdleInhibitor.
     * This method is supposed to be used when the connection to the Wayland
     * server goes away. If the connection is not valid anymore, it's not
     * possible to call release anymore as that calls into the Wayland
     * connection and the call would fail. This method cleans up the data, so
     * that the instance can be deleted or set up to a new zwp_idle_inhibitor_v1 interface
     * once there is a new connection available.
     *
     * It is suggested to connect this method to ConnectionThread::connectionDied:
     * @code
     * connect(connection, &ConnectionThread::connectionDied, idleinhibitor,
     * &IdleInhibitor::destroy);
     * @endcode
     *
     * @see release
     **/
    void destroy();

    operator zwp_idle_inhibitor_v1*();
    operator zwp_idle_inhibitor_v1*() const;

private:
    friend class IdleInhibitManager;
    explicit IdleInhibitor(QObject* parent = nullptr);
    class Private;
    std::unique_ptr<Private> d;
};

}
}

#endif
