"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CustomPanelsAdaptor = void 0;

var _uuid = require("uuid");

var _custom_panels = require("../../../common/constants/custom_panels");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class CustomPanelsAdaptor {
  constructor() {
    _defineProperty(this, "indexPanel", async function (client, panelBody) {
      try {
        const response = await client.callAsCurrentUser('observability.createObject', {
          body: {
            operationalPanel: panelBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Index Panel Error:' + error);
      }
    });

    _defineProperty(this, "updatePanel", async function (client, panelId, updatePanelBody) {
      try {
        const response = await client.callAsCurrentUser('observability.updateObjectById', {
          objectId: panelId,
          body: {
            operationalPanel: updatePanelBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Update Panel Error:' + error);
      }
    });

    _defineProperty(this, "getPanel", async function (client, panelId) {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: panelId
        });
        return response.observabilityObjectList[0];
      } catch (error) {
        throw new Error('Get Panel Error:' + error);
      }
    });

    _defineProperty(this, "viewPanelList", async function (client) {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'operationalPanel',
          maxItems: 10000
        });
        return response.observabilityObjectList.filter(panel => !panel.operationalPanel.applicationId).map(panel => ({
          name: panel.operationalPanel.name,
          id: panel.objectId,
          dateCreated: panel.createdTimeMs,
          dateModified: panel.lastUpdatedTimeMs
        }));
      } catch (error) {
        throw new Error('View Panel List Error:' + error);
      }
    });

    _defineProperty(this, "deletePanel", async function (client, panelId) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectById', {
          objectId: panelId
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Delete Panel Error:' + error);
      }
    });

    _defineProperty(this, "deletePanelList", async function (client, panelIdList) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectByIdList', {
          objectIdList: panelIdList
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Delete Panel List Error:' + error);
      }
    });

    _defineProperty(this, "createNewPanel", async (client, panelName, appId) => {
      const panelBody = {
        name: panelName,
        visualizations: [],
        timeRange: {
          to: 'now',
          from: 'now-1d'
        },
        queryFilter: {
          query: '',
          language: 'ppl'
        }
      };

      if (appId) {
        panelBody.applicationId = appId;
        panelBody.timeRange = {
          to: 'now',
          from: 'now-24h'
        };
      }

      try {
        const response = await this.indexPanel(client, panelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Create Panel Error:' + error);
      }
    });

    _defineProperty(this, "renamePanel", async (client, panelId, panelName) => {
      const updatePanelBody = {
        name: panelName
      };

      try {
        const response = await this.updatePanel(client, panelId, updatePanelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Rename Panel Error:' + error);
      }
    });

    _defineProperty(this, "clonePanel", async (client, panelId, panelName) => {
      const updatePanelBody = {
        name: panelName
      };

      try {
        const getPanel = await this.getPanel(client, panelId);
        const clonePanelBody = {
          name: panelName,
          visualizations: getPanel.operationalPanel.visualizations,
          timeRange: getPanel.operationalPanel.timeRange,
          queryFilter: getPanel.operationalPanel.queryFilter
        };
        const indexResponse = await this.indexPanel(client, clonePanelBody);
        const getClonedPanel = await this.getPanel(client, indexResponse.objectId);
        return {
          clonePanelId: getClonedPanel.objectId,
          dateCreated: getClonedPanel.createdTimeMs,
          dateModified: getClonedPanel.lastUpdatedTimeMs
        };
      } catch (error) {
        throw new Error('Clone Panel Error:' + error);
      }
    });

    _defineProperty(this, "addPanelFilter", async (client, panelId, query, language, to, from) => {
      const updatePanelBody = {
        timeRange: {
          to,
          from
        },
        queryFilter: {
          query,
          language
        }
      };

      try {
        const response = await this.updatePanel(client, panelId, updatePanelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Add Panel Filter Error:' + error);
      }
    });

    _defineProperty(this, "parseSavedVisualizations", visualization => {
      return {
        id: visualization.objectId,
        name: visualization.savedVisualization.name,
        query: visualization.savedVisualization.query,
        type: visualization.savedVisualization.type,
        timeField: visualization.savedVisualization.selected_timestamp.name,
        selected_date_range: visualization.savedVisualization.selected_date_range,
        selected_fields: visualization.savedVisualization.selected_fields,
        user_configs: visualization.savedVisualization.hasOwnProperty('user_configs') ? JSON.parse(visualization.savedVisualization.user_configs) : {},
        sub_type: visualization.savedVisualization.hasOwnProperty('sub_type') ? visualization.savedVisualization.sub_type : '',
        units_of_measure: visualization.savedVisualization.hasOwnProperty('units_of_measure') ? visualization.savedVisualization.units_of_measure : '',
        ...(visualization.savedVisualization.application_id ? {
          application_id: visualization.savedVisualization.application_id
        } : {})
      };
    });

    _defineProperty(this, "getAllSavedVisualizations", async client => {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'savedVisualization'
        });
        return response.observabilityObjectList.map(visualization => this.parseSavedVisualizations(visualization));
      } catch (error) {
        throw new Error('View Saved Visualizations Error:' + error);
      }
    });

    _defineProperty(this, "getSavedVisualizationById", async (client, savedVisualizationId) => {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: savedVisualizationId
        });
        const visualization = response.observabilityObjectList[0];
        return this.parseSavedVisualizations(visualization);
      } catch (error) {
        throw new Error('Fetch Saved Visualizations By Id Error:' + error);
      }
    });

    _defineProperty(this, "getVisualizations", async (client, panelId) => {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: panelId
        });
        return response.observabilityObjectList[0].operationalPanel.visualizations;
      } catch (error) {
        throw new Error('Get Visualizations Error:' + error);
      }
    });

    _defineProperty(this, "calculatOverlapArea", (bb1, bb2) => {
      const x_left = Math.max(bb1.x1, bb2.x1);
      const y_top = Math.max(bb1.y1, bb2.y1);
      const x_right = Math.min(bb1.x2, bb2.x2);
      const y_bottom = Math.min(bb1.y2, bb2.y2);
      if (x_right < x_left || y_bottom < y_top) return 0;
      return (x_right - x_left) * (y_bottom - y_top);
    });

    _defineProperty(this, "getTotalOverlapArea", panelVisualizations => {
      const newVizBox = {
        x1: 0,
        y1: 0,
        x2: 6,
        y2: 4
      };
      const currentVizBoxes = panelVisualizations.map(visualization => {
        return {
          x1: visualization.x,
          y1: visualization.y,
          x2: visualization.x + visualization.w,
          y2: visualization.y + visualization.h
        };
      });
      let isOverlapping = 0;
      currentVizBoxes.map(viz => {
        isOverlapping += this.calculatOverlapArea(viz, newVizBox);
      });
      return isOverlapping;
    });

    _defineProperty(this, "getNewVizDimensions", panelVisualizations => {
      let maxY = 0;
      let maxYH = 0; // check if we can place the new visualization at default location

      if (this.getTotalOverlapArea(panelVisualizations) === 0) {
        return {
          x: 0,
          y: 0,
          w: 6,
          h: 4
        };
      } // else place the new visualization at the bottom of the panel


      panelVisualizations.map(panelVisualization => {
        if (panelVisualization.y >= maxY) {
          maxY = panelVisualization.y;
          maxYH = panelVisualization.h;
        }
      });
      return {
        x: 0,
        y: maxY + maxYH,
        w: 6,
        h: 4
      };
    });

    _defineProperty(this, "addVisualization", async (client, panelId, savedVisualizationId, oldVisualizationId) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        let newDimensions;
        let visualizationsList = [];

        if (oldVisualizationId === undefined) {
          newDimensions = this.getNewVizDimensions(allPanelVisualizations);
          visualizationsList = allPanelVisualizations;
        } else {
          allPanelVisualizations.map(visualization => {
            if (visualization.id !== oldVisualizationId) {
              visualizationsList.push(visualization);
            } else {
              newDimensions = {
                x: visualization.x,
                y: visualization.y,
                w: visualization.w,
                h: visualization.h
              };
            }
          });
        }

        const newPanelVisualizations = [...visualizationsList, {
          id: 'panel_viz_' + (0, _uuid.v4)(),
          savedVisualizationId,
          ...newDimensions
        }];
        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: newPanelVisualizations
        });
        return newPanelVisualizations;
      } catch (error) {
        throw new Error('Add/Replace Visualization Error:' + error);
      }
    });

    _defineProperty(this, "addMultipleVisualizations", async (client, panelId, savedVisualizationIds) => {
      try {
        let allPanelVisualizations = await this.getVisualizations(client, panelId);
        let newDimensions;
        let visualizationsList = [...allPanelVisualizations];
        savedVisualizationIds.map(savedVisualizationId => {
          newDimensions = this.getNewVizDimensions(visualizationsList);
          visualizationsList = [...visualizationsList, {
            id: 'panel_viz_' + (0, _uuid.v4)(),
            savedVisualizationId,
            ...newDimensions
          }];
        });
        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: visualizationsList
        });
        return visualizationsList;
      } catch (error) {
        throw new Error('Add/Replace Visualization Error:' + error);
      }
    });

    _defineProperty(this, "editVisualization", async (client, panelId, visualizationParams) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        const filteredPanelVisualizations = [];

        for (let i = 0; i < allPanelVisualizations.length; i++) {
          for (let j = 0; j < visualizationParams.length; j++) {
            if (allPanelVisualizations[i].id === visualizationParams[j].i) {
              filteredPanelVisualizations.push({ ...allPanelVisualizations[i],
                x: visualizationParams[j].x,
                y: visualizationParams[j].y,
                w: visualizationParams[j].w,
                h: visualizationParams[j].h
              });
            }
          }
        }

        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: filteredPanelVisualizations
        });
        return filteredPanelVisualizations;
      } catch (error) {
        throw new Error('Edit Visualizations Error:' + error);
      }
    });

    _defineProperty(this, "addSamplePanels", async (client, savedVisualizationIds) => {
      try {
        const panelBody = (0, _custom_panels.createDemoPanel)(savedVisualizationIds);
        const indexResponse = await this.indexPanel(client, panelBody);
        const fetchPanel = await this.getPanel(client, indexResponse.objectId);
        const fetchResponse = {
          name: fetchPanel.operationalPanel.name,
          id: fetchPanel.objectId,
          dateCreated: fetchPanel.createdTimeMs,
          dateModified: fetchPanel.lastUpdatedTimeMs
        };
        return [fetchResponse];
      } catch (error) {
        throw new Error('Create Panel Error:' + error);
      }
    });
  }

}

exports.CustomPanelsAdaptor = CustomPanelsAdaptor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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