
=pod

=head1 COPYRIGHT

# (c) 1992-2023 Intel Corporation.                                              
# Intel, the Intel logo, Intel, MegaCore, NIOS II, Quartus and TalkBack         
# words and logos are trademarks of Intel Corporation or its                    
# subsidiaries in the U.S. and/or other countries. Other marks and              
# brands may be claimed as the property of others.                              
# See Trademarks on intel.com for full list of Intel trademarks or the          
# Trademarks & Brands Names Database (if Intel)                                 
# or See www.Intel.com/legal (if Altera)                                        
# Your use of Intel Corporation's design tools, logic functions and             
# other software and tools, and its AMPP partner logic functions, and           
# any output files any of the foregoing (including device programming           
# or simulation files), and any associated documentation or information         
# are expressly subject to the terms and conditions of the Altera               
# Program License Subscription Agreement, Intel MegaCore Function               
# License Agreement, or other applicable license agreement, including,          
# without limitation, that your use is for the sole purpose of                  
# programming logic devices manufactured by Intel and sold by Intel or          
# its authorized distributors.                                                  
# Please refer to the applicable agreement for further details.                 


=head1 NAME

acl::Encrypt.pm - Encrypt files under kernel_hdl

=head1 VERSION

$Header$

=head1 DESCRIPTION

This module provides the method to encrypt RTL files for 
default and simulation flows.

=cut

BEGIN {
  unshift @INC, (
    grep { -d $_ } (
      map { $ENV{INTELFPGAOCLSDKROOT} . $_ }
        qw(
        /host/windows64/bin/perl/lib/MSWin32-x64-multi-thread
        /host/windows64/bin/perl/lib
        /share/lib/perl
        /share/lib/perl/5.8.8 )
    )
  );
}




package acl::Encrypt;
use File::Find;
use File::Basename;
use File::Copy;
use strict;
use Exporter;
use acl::AOCDriverCommon;

our @ISA = qw(Exporter);
our @EXPORT = qw($encryption_mode $no_encryption $encryption_key $encryption_id $encryption_release_date);

#Holds all files that need to be encrypted
my @file_list = ();

#Variables for project directory and name
my $prj_dir;
my $base_name;
my $kernel_dir;
my $encrypt_dir;

#encryption controls
our $encryption_mode = 0;
our $no_encryption=0;

#encryption values
our $encryption_key = undef;
our $encryption_id = undef;
our $encryption_release_date = undef;

#Pushing full path of every FILE in kernel_hdl directory to the file_list array
sub get_files{
  if (-f $_) {
    my $full_path_file = $File::Find::name;
    push(@file_list, "${full_path_file}");
  }
}

#encrypt_1735 tool is used for encryption for any given simulator. The public keys for all vendors are publicly available and are hardcoded in the tool
sub sim_encrypt_file{
  my ($file) = @_;

  #Parsing file name, directory and extension
  my $label; 
  my $dir; 
  my $ext;
  ($label,$dir,$ext) = fileparse($file,'\..*');

  #encrypt_1735 command requires a different input and output file
  #Solution is to store the encrypted version in a seperate directory and move it back
  my $encrypted_file_path = "${encrypt_dir}/${label}${ext}";

  #qsys_generate command in create_design (aoc.pl) also creates a reference of each file in the ip directory
  my $ip_dir_file = "${prj_dir}/ip/kernel_system/kernel_system_${base_name}_sys/${base_name}_sys_140/sim/${label}${ext}";

  #Determining language from extension for encryption command
  #encrypt_1735 tool supports VHDL encryption but Ampcrypt does NOT. Therefore no point of having VHDL support for simulation only.
  my $language = ${ext} eq q(.sv) ? 'systemverilog' : ${ext} eq q(.v) ? 'verilog' : q();
  if($language eq q()){
    die("Kernel HDL must be system verilog or verilog.");
  }

  #encrypt_1735 command used for Mentor (modelsim) simulation
  my $sim_encrypt_command = "encrypt_1735 ${file} --language=${language} --simulation=mentor --of=${encrypted_file_path}";
  my $return_value = system($sim_encrypt_command);
  if ($return_value != 0) {
    die("Failed to execute \"$sim_encrypt_command\"");
  }

  #Copy the encrypted file to the <>.prj/ip/kernel_system/kernel_system_simple_sys directory
  copy $encrypted_file_path, $ip_dir_file;

  #Move the encrypted file back to the <>.prj/kernel_hdl/ directory
  move $encrypted_file_path, $file;
}

#ampcrypt tool is used for IP encryption (supported with Quartus) but does NOT work with simulation
sub ampcrypt_file{
  my ($file) = @_; 
    
  #Parsing file name, directory and extension
  my $label; 
  my $dir; 
  my $ext;
  ($label,$dir,$ext) = fileparse($file,'\..*');

  my $truncated_label = substr($label, 0, 63);

  #Ampcrypt libraries do not support VHDL file encryption 
  if($ext eq q(.vhd)){
    die("Ampcrypt libraries do not support the encryption of VHDL files");
  }

  #ampcrypt command requires a different input and output file
  #Solution is to store the encrypted version in a seperate directory and move it back
  my $encrypted_file_path="${encrypt_dir}/${label}${ext}";
    
  #ampcrypt command used for encryption 
  #encryption_key, encryption_id and encryption_release_date are set in acl::AOCInputParser::parse_args 
  #-3: triple-DES, -c: crack resistant, -i: input, -l: label, -o: output, -v: version, -k: 48 hex digit encryption key, -p: product/IP id
  my $amp_command = "ampcrypt -3 -c -i ${file} -l ${truncated_label} -o ${encrypted_file_path} -v ${encryption_release_date} -k ${encryption_key} -p ${encryption_id}";
    
  my $return_value = system($amp_command);
  if ($return_value != 0) {
    die("Failed to execute \"$amp_command\"");
  }

  #Move the encrypted file back to the <>.prj/kernel_hdl/ directory
  move $encrypted_file_path, $file;
}

sub encrypt_kernel{
  my ($final_work_dir, $base) = @_;

  my @encrypt_dirs = ();
  $prj_dir = $final_work_dir;
  $base_name = $base;
  $kernel_dir = "${prj_dir}/kernel_hdl";
  $encrypt_dir = "${prj_dir}/encrypted_rtl";

  #Traverse files recursively and do some processing on them (get_files function)
  find (\&get_files, $kernel_dir);

  #Create /encrypt directory in kernel_hdl
  #encrypt_1735 & ampcrypt tools require a different input and output file, therefore creating a directory to store encrypted versions
  mkdir("$encrypt_dir") or die "Couldn't create ${encrypt_dir} directory, $!";      

  #Iterate through all files that require encryption
  #Default flow and simulation flow use different encryption tools
  while(@file_list){
    my $next_file = shift @file_list;
    if ($new_sim_mode){
      sim_encrypt_file($next_file);
    }else{
      ampcrypt_file($next_file);
    }
  }
    
  #Remove the /encrypt directory in kernel_hdl
  rmdir($encrypt_dir) or die "Couldn't remove ${encrypt_dir} directory";

}
1;
